//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.5

import wesual.Create    1.0
import wesual.Ui         1.0
import wesual.Ui.Private 1.0

import wesual.data.Core  1.0

UiPopupAnchor {
    id : flyout

    signal productPicked(Product product)

    autoLayout : true

    popup : Rectangle {
        id : menu

        readonly property int preferredHeight :
            productView.contentHeight + filteredName.height + 15
        readonly property bool flipped : flyout.distanceBottom < 400
        readonly property bool shifted : flyout.distanceRight < width

        property int currentIndex : 0

        Keys.onPressed : {
            if (event.key === Qt.Key_PageUp || event.key === Qt.Key_PageDown) {
                // Do not propagate navigation keys while focus item
                event.accepted = true;
            }
        }
        Keys.onRightPressed : {}
        Keys.onLeftPressed  : {}
        Keys.onUpPressed    : {
            productView.currentIndex =
                    Math.max(0, productView.currentIndex - 1);
        }

        Keys.onDownPressed : {
            productView.currentIndex =
                    Math.min(productView.currentIndex + 1,
                             productSortedList.resultingDocuments.length - 1);
        }

        implicitWidth  : 302
        implicitHeight : preferredHeight

        color   : UiColors.getColor(UiColors.White)
        opacity : flyout.popupVisible ? 1 : 0
        enabled : flyout.popupVisible ? true : false

        border {
            width : 1
            color : UiColors.getColor(UiColors.ControlGrey20)
        }

        Behavior on opacity {
            NumberAnimation { duration : 200 }
        }

        UiFilterStringInput {
            id : filteredName

            anchors {
                top         : parent.top
                topMargin   : 5
                left        : parent.left
                leftMargin  : 5
                right       : parent.right
                rightMargin : 5
            }

            // Force focus on reopen
            Connections {
                target : flyout
                onAboutToShow : filteredName.forceActiveFocus()
                onAboutToHide : filteredName.clear()
                ignoreUnknownSignals : true
            }

            // Force focus on Open
            Component.onCompleted :  {
                filteredName.text = "";
                filteredName.forceActiveFocus();
            }

            Timer {
                id : timer

                interval : 500
                running  : false
                repeat   : false

                onTriggered :
                    productNameFilter.propertyFilter = filteredName.text
            }

            onTextChanged   : timer.restart()
            onInputAccepted : {
                var chosenProduct = productSortedList.resultingDocuments[
                            productView.currentIndex];
                flyout.productPicked(chosenProduct);
                flyout.hide();
            }
        }

        UiVerticalScrollView {
            anchors {
                left         : parent.left
                leftMargin   : 2
                rightMargin  : 2
                right        : parent.right
                top          : filteredName.bottom
                topMargin    : 4
                bottom       : parent.bottom
                bottomMargin : 5
            }

            ListView {
                id : productView

                model : productSortedList.resultingDocuments

                delegate : Item {
                    width  : parent ? parent.width : 100
                    height : 68

                    Rectangle {
                        id : selectionRect

                        anchors {
                            left        : parent.left
                            leftMargin  : 3
                            right       : parent.right
                            rightMargin : 3
                            top         : parent.top
                            bottom      : parent.bottom
                        }
                        color : UiColors.getColor(UiColors.White)

                        states : [
                            State {
                                name : "hover"
                                when : productView.currentIndex === index

                                PropertyChanges {
                                    target : selectionRect
                                    color  : UiColors.getColor(
                                                 UiColors.HoverGreen)
                                }
                            }

                        ]

                        AssetThumbnail {
                            id : image

                            asset  : modelData.primaryImage
                            width  : 60
                            height : 60

                            anchors {
                                left           : parent.left
                                leftMargin     : 6
                                verticalCenter : parent.verticalCenter
                            }
                        }

                        Column {
                            id : textColumn

                            anchors {
                                left       : image.right
                                leftMargin : 9
                                right      : parent.right
                                top        : parent.top
                                topMargin  : 5
                            }

                            UiText {
                                anchors {
                                    left  : parent.left
                                    right : parent.right
                                }

                                text     : modelData.name
                                color    : UiColors.getColor(UiColors.Black)
                                wrapMode : Text.Wrap

                                maximumLineCount : 2
                            }

                            PriceLabel {
                                font     : UiFonts.getFont(UiFonts.Regular, 12)
                                product  : modelData
                                color    : UiColors.getColor(UiColors.Black)

                                lineHeightMode : Text.FixedHeight
                                lineHeight     : 26
                            }
                        }

                        MouseArea {
                            id : selectionMouseArea

                            anchors.fill : parent
                            hoverEnabled : true

                            onContainsMouseChanged : {
                                if (containsMouse)
                                    productView.currentIndex = index;
                            }

                            onClicked : {
                                modelData.beginRecordChanges();
                                flyout.productPicked(modelData);
                                modelData.endRecordChanges();
                                flyout.hide();
                            }
                        }
                    }
                }
            }
        }
    }

    resources : [
        PersistentTypeListing {
            id : productListing

            type    : "product"
            context : Creator.context
        },
        DocumentListFilter {
            id : filteredProductListing

            documents : productListing.documents
            filters   : productNameFilter
        },
        StringFilter {
            id : productNameFilter

            property : "name"
        },
        DocumentListSorter {
            id : productSortedList

            documents : filteredProductListing.resultingDocuments
            sorters   : TypeSorter {
                property : "name"
            }
        }
    ]
}
